/*---------------------_---------------_--------------------_----_------    _
   ____ __  ___ ___ __| |___ _ __  ___| |_ ___ _ _  __ __ _(_)__| |__ _ ___| |_ 
  (_-< '_ \/ -_) -_) _` / _ \ '  \/ -_)  _/ -_) '_| \ V  V / / _` / _` / -_)  _|
  /__/ .__/\___\___\__,_\___/_|_|_\___|\__\___|_|    \_/\_/|_\__,_\__, \___|\__|
-----|_|----------------------------------------------------------|___/-  
    
  #HTML should look like this:
  
  <canvas width="300" height="300" id="speedometer_canvas"></canvas>
----------------------------------------------------------------------*/

//PUBLIC
/*--------------------------------------------------------------------*/
function Speedometer_Widget(canvas_id, level_text, num_text, path_to_widgets)
{
  this.canvas             = document.getElementById(canvas_id);
  this.context            = this.canvas.getContext('2d');
  this.background         = this.create_image(path_to_widgets + "speedometer/background.png");
  this.level_text         = level_text;
  this.num_text           = num_text;
  this.init();

  //this.canvas.ontouchstart  = this.onmousedown.bind(this);
  //this.canvas.ontouchend    = this.onmouseup.bind  (this);
  //this.canvas.ontouchleave  = this.onmouseup.bind  (this);
  //this.canvas.ontouchmove   = this.onmousemove.bind(this);
  
  //todo: why isn't this working in iOS (emulator)
  this.canvas.addEventListener("touchstart", this.onmousedown.bind(this), false);
  this.canvas.addEventListener("touchend"  , this.onmouseup.bind(this)  , false);
  this.canvas.addEventListener("touchleave", this.onmouseup.bind(this)  , false);
  this.canvas.addEventListener("touchmove ", this.onmousemove.bind(this), false);
  
  this.canvas.onmousedown = this.onmousedown.bind(this);
  //this.canvas.onmouseenter= this.onmousedown.bind(this);
  this.canvas.onmouseup   = this.onmouseup.bind  (this);
  this.canvas.onmouseout  = this.onmouseup.bind  (this);
  this.canvas.onmousemove = this.onmousemove.bind(this);
  this.interpolation_thread_run_loop_run();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.init = function()
{
  this.level_min     = 50; 
  this.level_max     = 250;
  this.num_min       = 0; 
  this.num_max       = 999;
  this.num_digits    = 0; //after decimal point
  this.level_digits  = 0; //after decimal point
  this.target_level  = 0;
  this.current_level = this.target_level;
  this.target_num    = 0;
  this.current_num   = this.target_num;
  this.target_goal   = this.level_max * 0.5;
  this.current_goal  = this.target_goal;
  this.goal_angle    = 0; //calculated when drawn for hit detection
  this.mouse_is_down = false;
  this.mouse_is_down_in_goal_indicator = false;
  this.interpolation_thread_runloop_duration = 75; //ms

  this.num_ticks     = 9;
  this.tick_spacing  = 2 * Math.PI / 13; //radians
  this.tick_start    = Math.floor(this.num_ticks * 0.5) * -this.tick_spacing;
  this.tick_end      = this.tick_start + this.tick_spacing * (this.num_ticks-1);
  this.clear();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.clear = function()
{
  this.set_level(this.level_min, true, this.num_min, true);
}

//PUBLIC
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.set_level_range = function(min, max)
{
  this.level_min = min; this.level_max = max;
  this.set_level(this.target_level, false, this.target_num, false);
  this.set_goal(this.target_goal, false);
}

//PUBLIC
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.set_num_range = function(min, max)
{
  this.num_min = min; this.num_max = max;
  this.set_level(this.target_level, false, this.target_num, false);
  this.set_goal(this.target_goal, false);
}

//PUBLIC
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.set_level_and_num_ranges = function(min, max)
{
  this.set_level_range(min, max);
  this.set_num_range(min, max);
}

//PUBLIC
/*--------------------------------------------------------------------*/
// the dial value is set independently of the center number.
// either can be set immediately, or interpolated
Speedometer_Widget.prototype.set_level = function(level, level_immediate, num, num_immediate)
{
  //interpolation_thread_run_loop_lock()
  this.target_level = this.clip(level, this.level_min, this.level_max);
  this.target_num   = this.clip(num  , this.num_min  , this.num_max  );
  
  if(level_immediate)
    this.current_level = this.target_level;
  if(num_immediate)
    this.current_num = this.target_num;    
  //interpolation_thread_run_loop_unlock()
}

//PUBLIC
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.set_goal = function(goal, immediate)
{
  //interpolation_thread_run_loop_lock()
  this.target_goal = this.clip(goal, this.level_min, this.level_max);  
  if(immediate)
    this.current_goal = this.target_goal;
  //interpolation_thread_run_loop_unlock()
}

//PRIVATE
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.draw = function(level, num, goal)
{
  var i;
  var red_line     = this.scalef(level, this.level_min, this.level_max, this.tick_start, this.tick_end);
  this.goal_angle  = this.scalef(goal , this.level_min, this.level_max, this.tick_start, this.tick_end);
  var big_radius   = 0.5 * Math.min(this.canvas.width, this.canvas.height); //to the edge of the drawing
  var radius       = 0.9 * big_radius; //to the edge of the clock
  this.context.clearRect(0, 0, this.canvas.width, this.canvas.height);
  
  this.context.drawImage(this.background, 0.5*this.canvas.width - radius, 0.5*this.canvas.height - radius, 2*radius, 2*radius); 

  this.context.textBaseline = "middle";
  this.context.textAlign    = "center"; 
  
  this.context.save();
  this.context.translate(0.5*this.canvas.width, 0.5*this.canvas.height);
  
  //draw ticks and numbers
  this.context.save();
  this.context.rotate(this.tick_start);
  this.context.font = "bold " + 0.1*radius + "px Arial";
  
  for(i=0; i<this.num_ticks; i++)
    {
      var n = this.scalef(i, 0, this.num_ticks-1, this.level_min, this.level_max);
      this.context.beginPath   ();
      this.context.moveTo      (0, 0.7 * -radius);
      this.context.lineTo      (0, 0.8 * -radius);
      this.context.lineWidth   = 6;
      this.context.strokeStyle = "rgb(0, 0, 0)";
      this.context.stroke      ();
      this.context.fillText    (Math.round(n), 0, -radius*0.6);
      this.context.rotate      (this.tick_spacing);
    }
  this.context.restore();
    
  //draw red line
  this.context.save        ();
  this.context.rotate      (red_line);
  this.context.beginPath   ();
  this.context.moveTo      (0, 0.4 * -radius);
  this.context.lineTo      (0, 0.8 * -radius);
  this.context.lineWidth   = 4;
  this.context.strokeStyle = "rgb(128, 0, 0)";
  this.context.stroke      ();
  this.context.restore     ();

  //draw goal indicator
  this.context.save        ();
  this.context.rotate      (this.goal_angle);

  this.context.lineWidth   = 4;
  this.context.strokeStyle = "rgb(252, 122, 29)";
  this.context.beginPath   ();
  this.context.arc         (0, 0, 0.9 * radius, 1.5*Math.PI - 0.8, 1.5*Math.PI + 0.8);
  this.context.stroke      ();
  
  this.context.translate   (0, 0.725 * -radius);
  this.context.fillStyle   = "rgb(252, 122, 29)";
  this.context.beginPath   ();
  this.context.arc         (0, 0, big_radius - (0.8 * radius), 1.5*Math.PI - 0.5, 1.5*Math.PI + 0.5);
  this.context.lineTo      (0, 0);
  this.context.fill        ();
  this.context.restore     ();
  
  
  //draw text
  this.context.restore     ();
  var font_size = 0.5*radius;
  this.context.font = "bold " + font_size + "px Arial";
  this.context.fillText(num.toFixed(this.num_digits), this.canvas.width*0.5, this.canvas.height*0.5);
  this.context.font = 0.25 * font_size + "px Arial";
  this.context.fillText(this.num_text, this.canvas.width*0.5, 0.47 * (this.canvas.height - font_size));
  this.context.font = 0.2 * font_size + "px Arial";
  this.context.fillText(level.toFixed(this.level_digits) + " " + this.level_text, this.canvas.width*0.5, 0.6 * (this.canvas.height + font_size));
}

//PRIVATE
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.onmousedown = function(e)
{ 
  this.mouse_is_down = true;
  var x = e.offsetX - this.canvas.offsetWidth * 0.5;
  var y = e.offsetY - this.canvas.offsetHeight * 0.5;
  var r = Math.sqrt(x*x + y*y);
  var big_radius   = 0.5 * Math.min(this.canvas.offsetWidth, this.canvas.offsetHeight);
    
  if(r > big_radius * 0.7)
    {
      var t = Math.atan2(y, x) + Math.PI * 0.5;
      if(t > Math.PI) t = t -= Math.PI*2;
      if((t > (this.goal_angle - 0.25)) &&  (t < (this.goal_angle + 0.25)))
      {
        this.mouse_is_down_in_goal_indicator = true;
        }
    }
}

//PRIVATE
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.onmouseup   = function(e)
{
  this.mouse_is_down_in_goal_indicator = false;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.onmousemove = function(e)
{
  if(this.mouse_is_down_in_goal_indicator)
  {
    var big_radius   = 0.5 * Math.min(this.canvas.offsetWidth, this.canvas.offsetHeight);
    var x = e.offsetX - this.canvas.offsetWidth * 0.5;
    var y = e.offsetY - this.canvas.offsetHeight * 0.5;
    var r = Math.sqrt(x*x + y*y);
    if(r > 0)
      {
        var t = Math.atan2(y, x) + Math.PI * 0.5;
        if(t > Math.PI) t = t -= Math.PI*2;
        var g  = this.scalef(t , this.tick_start, this.tick_end, this.level_min, this.level_max);
        this.set_goal(g, true);
      }
  }
}

//PRIVATE
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.scalef = function(x, in_min, in_max, out_min, out_max)
{
  return (x - in_min) * (out_max - out_min) / (in_max - in_min) + out_min;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.clip = function(x, min, max)
{
  return (x < min) ? min : (x > max) ? max : x;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.create_image = function(src)
{
  var img = document.createElement("IMG");
  img.src = src;
  img.style.display = "none";
  document.body.appendChild(img);
  return img;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Speedometer_Widget.prototype.interpolation_thread_run_loop_run = function()
{
  this.current_level = (0.8 * this.current_level) + (0.2 * this.target_level);
  this.current_num   = (0.8 * this.current_num  ) + (0.2 * this.target_num  );
  this.current_goal  = (0.8 * this.current_goal ) + (0.2 * this.target_goal );
  this.draw(this.current_level, this.current_num, this.current_goal);
  setTimeout(this.interpolation_thread_run_loop_run.bind(this), this.interpolation_thread_runloop_duration);
}



