/*-_----------_-_----------------_-_---_-------------------_----_-------   _
  | |__  __ _| | |  _ __  ___ __(_) |_(_)___ _ _   __ __ _(_)__| |__ _ ___| |_ 
  | '_ \/ _` | | | | '_ \/ _ (_-< |  _| / _ \ ' \  \ V  V / / _` / _` / -_)  _|
  |_.__/\__,_|_|_| | .__/\___/__/_|\__|_\___/_||_|  \_/\_/|_\__,_\__, \___|\__|
-------------------|_|-------------------------------------------|___/-- 
  #include <IC_LEVEL_2.js> //for statistics modules
  
  #HTML should look like this:
  <canvas width="300" height="300" id="ball_position_canvas"></canvas>
  <script>var Ball_Position_Cones_Widget = new Ball_Position_Cones_Widget('ball_position_canvas')</script>
----------------------------------------------------------------------*/

//PUBLIC
/*--------------------------------------------------------------------*/
function Ball_Position_Cones_Widget(canvas_id, path_to_widgets)
{
  this.cursor           = this.create_image(path_to_widgets + "ball_position/soccer_ball.png");
  this.cone             = this.create_image(path_to_widgets + "ball_position/cone.png");
  this.grass            = this.create_image(path_to_widgets + "ball_position/grass.png");
  this.canvas           = document.getElementById(canvas_id);
  this.context          = this.canvas.getContext('2d');
  
  this.init();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.init = function()
{
  //PUBLIC
  position                 = {x:0, x:0, z:0};
  
  //PUBLIC
  this.decimation_rate     = 10;   //save one every 10 samples
  this.N                   = 50;   //the filter order
  this.motion_threshold    = 0.01;
  
  //PUBLIC
  this.camera_should_pan   = true;
  this.camera_should_zoom  = true;
  this.drift_correction    = false;

  //PUBLIC
  this.min_zoom_width_2    = 1.5;  //half the canvas width in meters
  
  //PRIVATE
  this.canvas_width_2      = 0.5 * this.canvas.width;
  this.canvas_height_2     = 0.5 * this.canvas.height;

  //PRIVATE
  this.cursor_size_2       = 0.11; //half the cursor width in meters
  this.cone_size_2         = 0.11; //half the cone width in meters
  this.grass_size_2        = 0.5 ; //half the grass image tile width in meters
  
  this.cone_locations      = [{x:-1, y:1}, {x:1, y:1}, {x:1, y:-1}, {x:-1, y:-1}];
  
  this.avg_x_normal = new MovingAverage(this.N, 0, 0);
  this.avg_y_normal = new MovingAverage(this.N, 0, 0);
  this.avg_x_drift  = new MovingAverage(this.N, 0, 0);
  this.avg_y_drift  = new MovingAverage(this.N, 0, 0);
  
  //doing it twice helps eliminate gaps in repeating background image
  this.clear();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.clear = function()
{
  this.sample_counter      = 0;

  this.avg_x_normal.init(this.N, 0, 0);
  this.avg_y_normal.init(this.N, 0, 0);
  this.avg_x_drift.init (this.N, 0, 0);
  this.avg_y_drift.init (this.N, 0, 0);

  this.redraw_samples();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.set_N = function(N)
{
  this.avg_x_normal.N = N;
  this.avg_y_normal.N = N;
  this.avg_x_drift.N  = N;
  this.avg_y_drift.N  = N;
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.set_min_zoom_width = function(width_meters)
{
  this.min_zoom_width_2 = 0.5 * width_meters;
  this.redraw_samples();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.update = function(x, y, z)
{
  var did_add_sample = (this.sample_counter%this.decimation_rate) == 0;
  if(did_add_sample)
    {
      y = -y;
      var new_position = {x:x, y:y, z:z};
      var dist = this.motion_threshold;
      if(this.avg_x_normal.past_samples.length > 1)
        dist = this.get_distance(new_position, this.position);
      
      if((!this.drift_correction) || (dist >= this.motion_threshold))
        {
          this.avg_x_normal.update(x);
          this.avg_y_normal.update(y);
          this.avg_x_drift.update (x - this.avg_x_normal.mean);
          this.avg_y_drift.update (y - this.avg_y_normal.mean);
          this.position = new_position;
          this.redraw_samples();
        }
    }
  ++this.sample_counter;
  return did_add_sample;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.get_distance = function(a, b)
{
  var dx = a.x-b.x;
  var dy = a.y-b.y;
  var dz = a.z-b.z;
  return Math.sqrt(dx*dx + dy*dy + dz*dz);
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.redraw_samples = function()
{
  var i;
  var size, x1, x2=this.canvas_width_2, y1, y2=this.canvas_height_2;
  var zoom = 0;
  var samples_x = this.avg_x_normal.past_samples;
  var samples_y = this.avg_y_normal.past_samples;
  var mean_x    = this.avg_x_normal.mean;
  var mean_y    = this.avg_y_normal.mean;
  var var_x     = this.avg_x_normal.variance;
  var var_y     = this.avg_y_normal.variance;
 
  if(this.drift_correction)
    {
      samples_x = this.avg_x_drift.past_samples;
      samples_y = this.avg_y_drift.past_samples;
      mean_x    = this.avg_x_drift.mean;
      mean_y    = this.avg_y_drift.mean;
      var_x     = this.avg_x_drift.variance;
      var_y     = this.avg_y_drift.variance;
    }
   if(this.camera_should_zoom)
    {
      zoom = Math.max(var_x, var_y);
      zoom = Math.sqrt(zoom) * 3;
    }
    
  if(zoom < this.min_zoom_width_2) zoom = this.min_zoom_width_2;
  zoom = this.canvas_width_2 / zoom;


  if(!this.camera_should_pan)
    mean_x = mean_y = 0;
  
  //draw green background  
  this.clear_rect(mean_x, mean_y, zoom);
  
  
  //draw cones
  this.context.fillStyle   = "rgba(0, 0, 0, 255)";
  for(i=0; i<this.cone_locations.length; i++)
    this.draw_image_at_location_in_meters(this.cone, this.cone_locations[i].x, this.cone_locations[i].y, this.cone_size_2, this.cone_size_2, mean_x, mean_y, zoom);

  //draw tail
  this.context.lineWidth = 4;
  if(samples_x.length >= 1)
    {
      x1 = this.meters_to_pixels(samples_x[0], mean_x, zoom, this.canvas_width_2 );
      y1 = this.meters_to_pixels(samples_y[0], mean_y, zoom, this.canvas_height_2);
    }
  for(i=1; i<samples_x.length; i++)
    {
      x2 = this.meters_to_pixels(samples_x[i], mean_x, zoom, this.canvas_width_2 );
      y2 = this.meters_to_pixels(samples_y[i], mean_y, zoom, this.canvas_height_2);
      
      var lightness = i / (samples_x.length-1);
      this.context.strokeStyle = "rgba(255, 255, 255, "+lightness+")";
      this.draw_segment(x1, y1, x2, y2);
      x1 = x2;
      y1 = y2;
    }

  //draw ball
  size = zoom * this.cursor_size_2;
  x2 -= size;
  y2 -= size;
  size *= 2;
  this.context.drawImage(this.cursor, x2, y2, size, size);
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.draw_segment = function(start_x, start_y, end_x, end_y)
{
  this.context.beginPath(                );
  this.context.moveTo   (start_x, start_y);
  this.context.lineTo   (end_x  , end_y  );
  this.context.stroke   (                );
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.clear_rect = function(x_mean, y_mean, zoom)
{  
  var size = zoom * 2 * this.grass_size_2;
  var y = this.modulo(this.canvas_height_2 - zoom * (this.grass_size_2 + y_mean), size) - size;
  while(y < this.canvas.height)
    {
      var x = this.modulo(this.canvas_width_2 - zoom * (this.grass_size_2 + x_mean), size) - size;
      while(x < this.canvas.width)
        {
          this.context.drawImage(this.grass, x, y, size, size);
          x += size;
        }
      y += size;
    }
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.meters_to_pixels = function(meters, mean, zoom, canvas_size_2)
{
  return zoom * (meters - mean) + canvas_size_2;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.draw_image_at_location_in_meters = function(img, x, y, width_meters_2, height_meters_2, mean_x, mean_y, zoom)
{
  x = this.meters_to_pixels(x, mean_x, zoom, this.canvas_width_2 );
  y = this.meters_to_pixels(y, mean_y, zoom, this.canvas_height_2);
  size_x = zoom * width_meters_2;
  size_y = zoom * height_meters_2;
  x -= size_y;
  y -= size_x;
  this.context.drawImage(img, x, y, size_x*2, size_y*2);
}
      

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.modulo = function(a, b)
{
  if(a>0)
    return a % b;
  else 
    return b + (a % b);
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Cones_Widget.prototype.create_image = function(src)
{
  var img = document.createElement("IMG");
  img.src = src;
  img.style.display = "none";
  document.body.appendChild(img);
  return img;
}

