
Ball_Position_Widget_3D.grass_size_meters  = 2;    /* representational size of one grass tile */
Ball_Position_Widget_3D.ball_radius_meters = 0.11; /* representational size of one ball */
Ball_Position_Widget_3D.min_zoom_radius    = 6;   /* min representational size of the grass circle when zooming */

//PUBLIC
/*--------------------------------------------------------------------*/
function Ball_Position_Widget_3D(divname, path_to_widgets, draw_goal)
{
  this.decimation_interval = 0.1;   //update every 0.1 seconds
  this.running_duration      = 0;
  this.draw_goal = draw_goal;
  var div = document.getElementById(divname);
  
  this.W        = parseInt(div.style.width);
  this.H        = parseInt(div.style.height);
  
  
  this.scene    = new THREE.Scene();
  //this.camera   = new THREE.OrthographicCamera( -10, 10, 10, -10 , -500, 1000 );
  var aspect = this.W / this.H;
  this.camera   = new THREE.PerspectiveCamera(45/aspect, aspect, 1, 2000);
  
  this.scene = new THREE.Scene();

  this.ambient = new THREE.AmbientLight( 0x444444 );
  this.scene.add(this.ambient);

  this.directionalLight = new THREE.DirectionalLight(0xffeedd);
  this.directionalLight.position.set(100, 100, 0).normalize();
  this.scene.add(this.directionalLight);
  
  this.goal = null;
  this.soccer_ball    = null;
  this.ball_start_position = {x:0, y:Ball_Position_Widget_3D.ball_radius_meters, z:0};  
  
  // Goal (Clara.io loader example)
  if(draw_goal)
    {
      var goal_object_loader = new THREE.ObjectLoader();
      goal_object_loader.load(path_to_widgets + "ball_position_3d/3d_models/goal.json", 
        function ( _obj ) 
          {
            console.log(_obj);
            this.scene.add( _obj );
            this.goal = _obj;
          }.bind(this) );
    }
    
  /* Ball Path Line */
  this.MAX_LINE_POINTS = 500;
  var line_geometry = new THREE.BufferGeometry();
  var line_positions = new Float32Array( this.MAX_LINE_POINTS * 3 ); // 3 vertices per point
  line_geometry.addAttribute( 'position', new THREE.BufferAttribute( line_positions, 3 ) );
  var line_material = new THREE.LineBasicMaterial( { color: 0xFFFFFF } );
  this.line = new THREE.Line( line_geometry,  line_material );
  this.scene.add( this.line );

  /* Grass */
  var grass_geometry = new THREE.CircleGeometry(1, 64);
  var grass_texture = new THREE.TextureLoader().load( path_to_widgets + "ball_position_3d/3d_models/grass.png" );
  grass_texture.wrapS = THREE.RepeatWrapping;
  grass_texture.wrapT = THREE.RepeatWrapping;
  grass_texture.repeat.set(Ball_Position_Widget_3D.grass_size_meters, Ball_Position_Widget_3D.grass_size_meters);
  var grass_material = new THREE.MeshLambertMaterial({map: grass_texture});
  this.plane = new THREE.Mesh( grass_geometry, grass_material );
  this.plane.rotation.x = -Math.PI / 2;
  this.scene.add( this.plane );
  
    /*Soccer Ball*/
  var ball_geometry = new THREE.SphereGeometry(Ball_Position_Widget_3D.ball_radius_meters, 15, 15 );
  var ball_texture = new THREE.TextureLoader().load( path_to_widgets + "ball_position_3d/ball_3.png");
  ball_texture.wrapS = THREE.RepeatWrapping;
  ball_texture.wrapT = THREE.RepeatWrapping;
  ball_texture.repeat.set(1, 1);
  var ball_material = new THREE.MeshLambertMaterial({map: ball_texture});
  this.soccer_ball = new THREE.Mesh( ball_geometry, ball_material );
  this.soccer_ball.position.x = this.ball_start_position.x;
  this.soccer_ball.position.y = this.ball_start_position.y;
  this.soccer_ball.position.z = this.ball_start_position.z;
  this.scene.add( this.soccer_ball );

  /* circular grid lines */
  var grid_ellipse  = new THREE.EllipseCurve(0, 0, 1, 1, 0, 2.0 * Math.PI, false); //(Center_Xpos, Center_Ypos, Xradius, Yradius, StartAngle, EndAngle, isClockwise)
  var grid_path     = new THREE.CurvePath();
  grid_path.add(grid_ellipse);
  this.grid_geometery = grid_path.createPointsGeometry(100);
  this.grid_material = new THREE.LineBasicMaterial({color:0x99AA99, opacity:1.0});

  this.renderer = new THREE.WebGLRenderer();
  this.renderer.setPixelRatio(window.devicePixelRatio);
  this.renderer.setSize(this.W, this.H);

  /* user interface */  
  this.div = document.getElementById(divname);
  this.div.appendChild( this.renderer.domElement );		
  this.div.onmousedown = this.onmousedown.bind(this);
  this.div.onmouseup   = this.onmouseup.bind  (this);
  this.div.onmouseout  = this.onmouseup.bind  (this);
  this.div.onmousemove = this.onmousemove.bind(this);
  this.mouse_is_dragging = false;

  /* camera */
  this.camera.position.x = 28.28;
  this.camera.position.y = 10;
  this.camera.position.z = -28.28;
  this.camera.lookAt(  new THREE.Vector3( 0, 0, 0 ) );
  
  this.did_intersect_goal_plane = false;
  this.scored_goal_message = "too low";
  
  //this.init();
  
  this.render_run_loop_run();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.init = function()
{
  this.ball_start_position.x = 0;
  this.ball_start_position.y = Ball_Position_Widget_3D.ball_radius_meters;
  this.ball_start_position.z = 0;

  this.clear();
}


//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.clear = function()
{
  if(this.soccer_ball == null) return;
  this.running_duration      = 0;
  
  if(this.soccer_ball != null)
    {
      this.soccer_ball.position.x = this.ball_start_position.x;
      this.soccer_ball.position.y = this.ball_start_position.y;
      this.soccer_ball.position.z = this.ball_start_position.z;
    }
    
  this.did_intersect_goal_plane = false;
  this.scored_goal_message = "too low";
  
  var line_positions = this.line.geometry.attributes.position.array;
  for(var i=0; i<this.MAX_LINE_POINTS*3; i+=3)
  {
    line_positions[i+0] = this.soccer_ball.position.x;
    line_positions[i+1] = this.soccer_ball.position.y;
    line_positions[i+2] = this.soccer_ball.position.z;    
  }
  this.line.geometry.attributes.position.needsUpdate = true;
  this.set_zoom(Ball_Position_Widget_3D.min_zoom_radius);
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.goal_scored_message = function()
{
  return this.scored_goal_message;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.set_zoom = function(ground_radius)
{
  if(ground_radius < Ball_Position_Widget_3D.min_zoom_radius)
    ground_radius = Ball_Position_Widget_3D.min_zoom_radius;
    
  this.plane.scale.x = this.plane.scale.y = ground_radius;
  var tile_size = ground_radius / Ball_Position_Widget_3D.grass_size_meters;
  this.plane.material.map.repeat.set(tile_size, tile_size);
  this.plane.material.map.offset.x = -0.5 * tile_size;
  if(!this.draw_goal)
    this.plane.material.map.offset.y = this.plane.material.map.offset.x;
  this.plane.material.map.needsUpdate = true;
  
  var camera = this.get_camera_rho_theta();
  camera.rho = 40 * ground_radius / Math.tan(0.5 * this.camera.fov);
  this.set_camera_rho_theta(camera);
  this.camera.position.y = camera.rho * 0.25;
  this.camera.lookAt(  new THREE.Vector3( 0, 0, 0 ) );
  
  if(this.draw_goal)
    this.plane.position.z = -(ground_radius - Ball_Position_Widget_3D.min_zoom_radius);

  else{
  var circle;
  while((circle = this.scene.getObjectByName("circle")))
    this.scene.remove(circle);
    
  for(var i=2; i<=ground_radius; i+=2)
    {
      var grid_line = new THREE.Line(this.grid_geometery, this.grid_material);
      grid_line.name = "circle";
      grid_line.rotation.x = -Math.PI / 2;
      grid_line.scale.x = i;
      grid_line.scale.y = i;
      this.scene.add( grid_line );
    }
  }
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.draw_ideal_trajectory = function(start_x, start_y, end_x, end_y, height)
{
  var dist_x = end_x - start_x;
  var dist_y = end_y - start_y;
  var dist   = Math.hypot(dist_x, dist_y);
  var middle = 0.5*dist;

  c = -height / (middle*middle);
  
  this.clear();
  var i;
  var N = this.MAX_LINE_POINTS;
  
  for(i=0; i<N; i++)
    {
      var _x = dist * i/N;
      var x = start_x + _x * dist_x / dist;
      var y = start_y + _x * dist_y / dist;
      var z = _x - middle;
      z *= z;
      z *= c;
      z += height;
      
      //this is inneficient because it redraws all the line segments each time...
      this.update(this.decimation_interval, x, y, z);
    }
}

/*--------------------------------------------------------------------*/
///* returns object of type {rho:r, theta:t} */
Ball_Position_Widget_3D.prototype.get_camera_rho_theta = function()
{
  var t = Math.atan2(this.camera.position.z, this.camera.position.x);
  var r   = Math.sqrt(this.camera.position.z*this.camera.position.z + this.camera.position.x*this.camera.position.x);
  
  return {rho:r, theta:t};
}

/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.set_camera_rho_theta = function(a /* object of type {rho:r, theta:t} */)
{
  this.camera.position.x = a.rho * Math.cos(a.theta);
  this.camera.position.z = a.rho * Math.sin(a.theta);
  this.camera.lookAt(  new THREE.Vector3( 0, 0, 0 ) );
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.update = function(sample_interval, x, y, z)
{
/*
  x = parseFloat(x);
  y = parseFloat(y);
  z = parseFloat(z);
*/
  var did_add_sample = false;
  this.running_duration += sample_interval;
  
  if(this.running_duration >= this.decimation_interval)
    {
      did_add_sample = true;
      this.running_duration -= this.decimation_interval;
    }
  
  if(did_add_sample)
    {  
      z = (z<0.0) ? 0.0 : z;
      this.soccer_ball.position.x = this.ball_start_position.x + y;
      this.soccer_ball.position.y = this.ball_start_position.y + z;
      this.soccer_ball.position.z = this.ball_start_position.z + x;
   
      var positions = this.line.geometry.attributes.position.array;
      
      //cannot push and pop, cannot dynamically add segments, cannot memmove(), but there must be a better way...
      for(var i=0; i<(this.MAX_LINE_POINTS-1)*3; i+=3)
        {
          positions[i+0] = positions[i+3];
          positions[i+1] = positions[i+4];
          positions[i+2] = positions[i+5];
        }
  
      positions[(this.MAX_LINE_POINTS-1)*3 + 0] = this.soccer_ball.position.x;
      positions[(this.MAX_LINE_POINTS-1)*3 + 1] = this.soccer_ball.position.y;
      positions[(this.MAX_LINE_POINTS-1)*3 + 2] = this.soccer_ball.position.z;
      
      var prev_position = 
      {
        x: positions[(this.MAX_LINE_POINTS-2)*3 + 0],
        y: positions[(this.MAX_LINE_POINTS-2)*3 + 1],
        z: positions[(this.MAX_LINE_POINTS-2)*3 + 2]
      }
      
      this.check_goal_scored(this.soccer_ball.position, prev_position);
      
      this.line.geometry.attributes.position.needsUpdate = true;
      
      this.set_zoom(1.1 * Math.hypot(this.soccer_ball.position.x, this.soccer_ball.position.z));
    }
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.onmousedown = function(e) 
{
  this.mouse_is_dragging = true;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.onmouseup = function(e)
{
  this.mouse_is_dragging = false;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.onmousemove = function(e) 
{
  if(this.mouse_is_dragging)
    {
      var camera = this.get_camera_rho_theta();
      camera.theta += e.movementX * 0.01;
      this.set_camera_rho_theta(camera);
    }
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.render_run_loop_run = function() 
{
	requestAnimationFrame( this.render_run_loop_run.bind(this) );
	this.renderer.render( this.scene, this.camera );
}

/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.check_goal_scored = function(a /* segment start {x:x, y:y, z:z} */, b /* segment end {x:x, y:y, z:z} */)
{
  if(this.did_intersect_goal_plane)
    return;

  var intersection = this.check_ball_segment_intersects_goal_plane(a, b);

  if(intersection != false)
    {
      this.did_intersect_goal_plane = true;
      var result;
      
      if(intersection.y > 2.44)
        result = "Too high";
      else if (intersection.x > 3.66)
        result = "Too far right";
      else if (intersection.x < -3.66)
        result = "Too far left";
      else 
        result = "Goal!";
        
      this.scored_goal_message = result;
    }
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget_3D.prototype.check_ball_segment_intersects_goal_plane = function(a /* segment start {x:x, y:y, z:z} */, b /* segment end {x:x, y:y, z:z} */)
{
  var any_point_on_goal_plane = new Quaternion(0, 0, 0, 0);
  var goal_plane_normal = new Quaternion(0, 0, 0, 1);
  var start = new Quaternion(0, a.x, a.y, a.z);
  var length = new Quaternion(0, b.x-a.x, b.y-a.y, b.z-a.z);
  
  var d = goal_plane_normal.vector_dot_product(any_point_on_goal_plane);
  var dot_normal_length = goal_plane_normal.vector_dot_product(length);
  
  //segment is parellel to goal plane
  if(dot_normal_length == 0)
    return false;
  
  var x = (d - goal_plane_normal.vector_dot_product(start)) / dot_normal_length;
  length.normalize(length);
  length.multiply_real(x, length);
  length.add(start, length);
  
  if (x >= 0.0 && x <= 1.0)
    return {x: length.q[1], y: length.q[2], z: length.q[3]};
  
  return false;
}
