/*-_----------_-_----------------_-_---_-------------------_----_-------   _
  | |__  __ _| | |  _ __  ___ __(_) |_(_)___ _ _   __ __ _(_)__| |__ _ ___| |_ 
  | '_ \/ _` | | | | '_ \/ _ (_-< |  _| / _ \ ' \  \ V  V / / _` / _` / -_)  _|
  |_.__/\__,_|_|_| | .__/\___/__/_|\__|_\___/_||_|  \_/\_/|_\__,_\__, \___|\__|
-------------------|_|-------------------------------------------|___/-- 
  #include <IC_LEVEL_2.js> //for statistics modules
  
  #HTML should look like this:
  <canvas width="300" height="300" id="ball_position_canvas"></canvas>
  <script>var ball_position_widget = new Ball_Position_Widget('ball_position_canvas')</script>
----------------------------------------------------------------------*/

//PUBLIC
/*--------------------------------------------------------------------*/
function Ball_Position_Widget(canvas_id, path_to_widgets)
{
  this.cursor           = this.create_image(path_to_widgets + "ball_position/soccer_ball.png");
  this.cones            = this.create_image(path_to_widgets + "ball_position/cones.png");
  this.grass            = this.create_image(path_to_widgets + "ball_position/grass.png");
  this.canvas           = document.getElementById(canvas_id);
  this.context          = this.canvas.getContext('2d');
  
  this.init();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.init = function()
{
  //PUBLIC
  this.decimation_rate     = 10;   //save one every 10 samples
  this.num_samples         = 50;   //the filter order
  this.motion_threshold    = 0.01;
  
  //PUBLIC
  this.camera_should_pan   = true;
  this.camera_should_zoom  = true;
  this.drift_correction    = false;

  //PUBLIC
  this.min_zoom_width_2    = 1.5;  //half the canvas width in meters
  
  //PRIVATE
  this.canvas_width_2      = 0.5 * this.canvas.width;
  this.canvas_height_2     = 0.5 * this.canvas.height;

  //PRIVATE
  this.cursor_size_2       = 0.11; //half the cursor width in meters
  this.cones_size_2        = 0.5;  //half the cone spacing in meters
  
  //doing it twice helps eliminate gaps in repeating background image
  this.clear();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.clear = function()
{
  this.samples             = new Array();
  this.filtered_samples    = new Array();
  this.sample_counter      = 0;
  this.normalization_x     = new OnlineAverage();
  this.normalization_y     = new OnlineAverage();
  this.prev                = {x:0, y:0, z:0};

  this.redraw_samples();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.set_min_zoom_width = function(width_meters)
{
  this.min_zoom_width_2 = 0.5 * width_meters;
  //this.redraw_samples();
}

//PUBLIC
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.update = function(x, y, z)
{
  if(this.sample_counter%this.decimation_rate == 0)
    {
      y = -y;
      var e = {x:x, y:y, z:z};
      var dist = this.motion_threshold;
      if (this.samples.length > 1)
        dist = this.get_distance(e, this.samples[this.samples.length-1]);

      if((!this.drift_correction) || (dist >= this.motion_threshold))
        {
          this.samples.push(e);
          this.filtered_samples.push({x:x, y:y, z:z});
          while(this.samples.length > this.num_samples)
            {
              this.samples.shift();
              this.filtered_samples.shift();
            }
          this.redraw_samples();
        }
    }
  ++this.sample_counter;
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.get_distance = function(a, b)
{
  var dx = a.x-b.x;
  var dy = a.y-b.y;
  var dz = a.z-b.z;
  return Math.sqrt(dx*dx + dy*dy + dz*dz);
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.redraw_samples = function()
{
  var x1=0, x2=0, y1=0, y2=0;
  var zoom = 0;
  var samples = this.samples;
  var size;
  
  this.get_statistics(this.samples, this.normalization_x, this.normalization_y);
  if(this.filtered_samples.length > 0)
    {
      this.filtered_samples[this.filtered_samples.length-1].x -= this.normalization_x.mean;
      this.filtered_samples[this.filtered_samples.length-1].y -= this.normalization_y.mean;
    }
  if(this.drift_correction)
    {
      this.get_statistics(this.filtered_samples, this.normalization_x, this.normalization_y);
      samples = this.filtered_samples;
    }
  
  if(this.camera_should_zoom)
    {
      zoom = Math.max(this.normalization_x.variance, this.normalization_y.variance);
      zoom = Math.sqrt(zoom) * 3;
    }
  if(zoom < this.min_zoom_width_2) zoom = this.min_zoom_width_2;
  zoom = this.canvas_width_2 / zoom;

  if(!this.camera_should_pan)
    this.normalization_x.mean = this.normalization_y.mean = 0;
  
  //draw green background  
  this.clear_rect(this.normalization_x.mean, this.normalization_y.mean, zoom);
  
  //draw cones
  x1 = this.canvas_width_2  - zoom * (this.cones_size_2 + this.normalization_x.mean);
  y1 = this.canvas_height_2 - zoom * (this.cones_size_2 + this.normalization_y.mean);
  size = zoom * 2 * this.cones_size_2;
  
  this.context.drawImage(this.cones, x1, y1, size, size); 

  //draw tail
  this.context.lineWidth = 4;
  var i;
  for(i=0; i<samples.length-1; i++)
    {
      x1 = zoom * (samples[i  ].x - this.normalization_x.mean);
      x2 = zoom * (samples[i+1].x - this.normalization_x.mean);
      y1 = zoom * (samples[i  ].y - this.normalization_y.mean);
      y2 = zoom * (samples[i+1].y - this.normalization_y.mean);
      var lightness = i / (samples.length-1);
      this.context.strokeStyle = "rgba(255, 255, 255, "+lightness+")";
      this.draw_segment(x1, y1, x2, y2);
    }

  //draw ball
  size = zoom * 2 * this.cursor_size_2;
  x2 -= 0.5 * size;
  y2 -= 0.5 * size;
  this.context.drawImage(this.cursor, this.canvas_width_2+x2, this.canvas_height_2+y2, size, size);
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.get_statistics = function(samples, norm_x, norm_y)
{
  //todo: this could be way more efficient; We don't really need
  //to calculate the mean and std dev again from sctarch.
  //we can just subtract out (oldest_sample / N) and add in 
  //(newest_sample / N).
  norm_x.init();
  norm_y.init();
  var i;
  for(i=0; i<samples.length; i++)
    {
      norm_x.update(samples[i].x);
      norm_y.update(samples[i].y);
    }
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.draw_segment = function(start_x, start_y, end_x, end_y)
{
  start_x += this.canvas_width_2 ;
  start_y += this.canvas_height_2;
  end_x   += this.canvas_width_2 ;
  end_y   += this.canvas_height_2;
  this.context.beginPath(                );
  this.context.moveTo   (start_x, start_y);
  this.context.lineTo   (end_x  , end_y  );
  this.context.stroke   (                );
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.clear_rect = function(x_mean, y_mean, zoom)
{  
  var size = zoom * 2 * this.cones_size_2;
  
  var y = this.modulo(this.canvas_height_2 - zoom * (this.cones_size_2 + y_mean), size) - size;
  while(y < this.canvas.height)
    {
      var x = this.modulo(this.canvas_width_2 - zoom * (this.cones_size_2 + x_mean), size) - size;
      while(x < this.canvas.width)
        {
          this.context.drawImage(this.grass, x, y, size, size);
          x += size;
        }
      y += size;
    }
     /*
  this.context.beginPath();
  this.grass_pattern    = this.context.createPattern(this.grass, "repeat");
  this.context.fillStyle = this.grass_pattern;
  //this.context.fillStyle = "rgb(64, 128, 0)"
  this.context.rect(0, 0, this.canvas.width, this.canvas.height);
  this.context.fill();
  */
}

/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.modulo = function(a, b)
{
  if(a>0)
    return a % b;
  else 
    return b + (a % b);
}

//PRIVATE
/*--------------------------------------------------------------------*/
Ball_Position_Widget.prototype.create_image = function(src)
{
  var img = document.createElement("IMG");
  img.src = src;
  img.style.display = "none";
  document.body.appendChild(img);
  return img;
}

